#region References

using System;
using System.Collections;
using System.ComponentModel;
using System.ComponentModel.Design;
using System.Drawing;
using System.Data;
using System.Windows.Forms;
using System.Configuration;
using System.Text;
using System.Text.RegularExpressions;

using gov.va.med.vbecs.BOL;
using gov.va.med.vbecs.DAL.VAL;
using gov.va.med.vbecs.DAL.HL7AL;

using TABLE = gov.va.med.vbecs.Common.VbecsTables;

#endregion

namespace gov.va.med.vbecs.GUI.controls
{
	#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///This GUI User Control allows users to select a patient via (partial) last name, full ssn, last 4 of ssn, last 5 (intial + last 4 ssn), or accession number
		///</summary>

		#endregion

	public class PatientSelectTool : GUI.controls.BaseControl
	{
		#region Variables

		//
		private const string NEWLINE = "\n"; 

		private const string TOOLTIP_BTN_SEARCH = "Enter Full Name, Full Patient ID, Full Last Name, Last 4 Patient ID, or First Initial of Last Name + Last 4 Patient ID.";
		private const string TOOLTIP_LV_PATIENTS = "Click to choose a patient from the list.";						
		//
		// CR 2522: Room and Bed are now stored in one field
		private const string TOOLTIP_TXT_PATIENT_LOCATION = "Patient's current location. Point of Care, Room and Bed and Division.";		
		private const string TOOLTIP_TXT_SEARCH_STRING = "Enter Last Name; Full,  Last 4 or Last 5 SSN; or Accession No. to search.";		

		private System.EventHandler onSIAndTRClick;
		private System.EventHandler onBloodAvailClick;
		private System.EventHandler onVbecsInfoClick;
		private System.EventHandler onMedicationsClick;
		private System.EventHandler onTransHistoryClick;
		private System.EventHandler onClinicalInfoClick;
		//
		private BOL.Patient patient;
		private System.Data.DataRow _dataRow;
		private DataTable _vbecsPatients;
		private DataTable _vistaPatients;

		/// <summary>
		/// Public event raised when a patient has been selected in the control
		/// </summary>
		public event EventHandler PatientSelected;

		/// <summary>
		/// Public event raised when no patient has been selected in the control
		/// </summary>
		public event EventHandler PatientDeselected;
		
		private Regex _uidEx;
		private Regex _lastfiveEx;
		private Regex _ssnEx;
		private Regex _nameEx;

		#endregion
	
		#region Form Variables
		
		private System.Windows.Forms.GroupBox grpImage;
		private System.Windows.Forms.PictureBox pictureBox1;
		private System.Windows.Forms.Label lblSearchString;
		private gov.va.med.vbecs.GUI.controls.PagingListView lvPatients;
		private System.Windows.Forms.ColumnHeader columnHeader1;
		private System.Windows.Forms.ColumnHeader columnHeader2;
		private System.Windows.Forms.ColumnHeader columnHeader3;
		private System.Windows.Forms.GroupBox grpSelected;
		private System.Windows.Forms.TextBox txtPatientName;
		private System.Windows.Forms.Label lblPatientName;
		private System.Windows.Forms.Label lblABORh;
		private System.Windows.Forms.TextBox txtABORh;
		private System.Windows.Forms.Label lblSSN;
		private System.Windows.Forms.Label lblDOB;
		private System.Windows.Forms.Label lblSex;
		private System.Windows.Forms.TextBox txtSSN;
		private System.Windows.Forms.TextBox txtDOB;
		private System.Windows.Forms.TextBox txtSex;
		private System.Windows.Forms.ToolTip toolTip1;
		private System.Windows.Forms.GroupBox grpSearch;
		private gov.va.med.vbecs.GUI.controls.SpecialInstructionsDisplayPanel specialInstructionsDisplayPanel1;
		private gov.va.med.vbecs.GUI.controls.PatientInfoToolbar patientInfoToolbar1;
		private System.Windows.Forms.Splitter splitter1;
		private System.Windows.Forms.Panel panel1;
		private gov.va.med.vbecs.GUI.controls.VbecsButton btnSearch;
		private System.ComponentModel.IContainer components;
		private System.Windows.Forms.PictureBox picJustified;
		private System.Windows.Forms.PictureBox pictureBox2;
		private gov.va.med.vbecs.GUI.controls.BarcodeTextbox txtSearchString;
		private System.Windows.Forms.TextBox txtCompatibbility;
		private System.Windows.Forms.Label lblCompatibility;
		private System.Windows.Forms.Label lblPercentage;
		private System.Windows.Forms.HelpProvider hlpHelpChm;
		private System.Windows.Forms.TextBox txtPatientLocation;
		private System.Windows.Forms.Label label1;
		private bool _searchVistALink = false;

		#endregion

		#region Component Designer generated code
		/// <summary> 
		/// Required method for Designer support - do not modify 
		/// the contents of this method with the code editor.
		/// </summary>
		private void InitializeComponent()
		{
			this.components = new System.ComponentModel.Container();
			System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(PatientSelectTool));
			this.grpImage = new System.Windows.Forms.GroupBox();
			this.pictureBox1 = new System.Windows.Forms.PictureBox();
			this.grpSearch = new System.Windows.Forms.GroupBox();
			this.pictureBox2 = new System.Windows.Forms.PictureBox();
			this.btnSearch = new gov.va.med.vbecs.GUI.controls.VbecsButton();
			this.lblSearchString = new System.Windows.Forms.Label();
			this.lvPatients = new gov.va.med.vbecs.GUI.controls.PagingListView();
			this.columnHeader1 = new System.Windows.Forms.ColumnHeader();
			this.columnHeader2 = new System.Windows.Forms.ColumnHeader();
			this.columnHeader3 = new System.Windows.Forms.ColumnHeader();
			this.txtSearchString = new gov.va.med.vbecs.GUI.controls.BarcodeTextbox(this.components);
			this.grpSelected = new System.Windows.Forms.GroupBox();
			this.label1 = new System.Windows.Forms.Label();
			this.txtPatientLocation = new System.Windows.Forms.TextBox();
			this.lblPercentage = new System.Windows.Forms.Label();
			this.lblCompatibility = new System.Windows.Forms.Label();
			this.txtCompatibbility = new System.Windows.Forms.TextBox();
			this.picJustified = new System.Windows.Forms.PictureBox();
			this.patientInfoToolbar1 = new gov.va.med.vbecs.GUI.controls.PatientInfoToolbar();
			this.specialInstructionsDisplayPanel1 = new gov.va.med.vbecs.GUI.controls.SpecialInstructionsDisplayPanel();
			this.txtPatientName = new System.Windows.Forms.TextBox();
			this.lblPatientName = new System.Windows.Forms.Label();
			this.txtSSN = new System.Windows.Forms.TextBox();
			this.txtABORh = new System.Windows.Forms.TextBox();
			this.txtSex = new System.Windows.Forms.TextBox();
			this.lblDOB = new System.Windows.Forms.Label();
			this.txtDOB = new System.Windows.Forms.TextBox();
			this.lblSSN = new System.Windows.Forms.Label();
			this.lblSex = new System.Windows.Forms.Label();
			this.lblABORh = new System.Windows.Forms.Label();
			this.toolTip1 = new System.Windows.Forms.ToolTip(this.components);
			this.splitter1 = new System.Windows.Forms.Splitter();
			this.panel1 = new System.Windows.Forms.Panel();
			this.hlpHelpChm = new System.Windows.Forms.HelpProvider();
			this.grpImage.SuspendLayout();
			this.grpSearch.SuspendLayout();
			this.grpSelected.SuspendLayout();
			this.SuspendLayout();
			// 
			// grpImage
			// 
			this.grpImage.BackColor = System.Drawing.SystemColors.Control;
			this.grpImage.Controls.Add(this.pictureBox1);
			this.grpImage.Dock = System.Windows.Forms.DockStyle.Left;
			this.grpImage.Location = new System.Drawing.Point(0, 0);
			this.grpImage.Name = "grpImage";
			this.grpImage.Size = new System.Drawing.Size(52, 400);
			this.grpImage.TabIndex = 3;
			this.grpImage.TabStop = false;
			// 
			// pictureBox1
			// 
			this.pictureBox1.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox1.Image")));
			this.pictureBox1.Location = new System.Drawing.Point(4, 12);
			this.pictureBox1.Name = "pictureBox1";
			this.pictureBox1.Size = new System.Drawing.Size(45, 45);
			this.pictureBox1.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.pictureBox1.TabIndex = 38;
			this.pictureBox1.TabStop = false;
			// 
			// grpSearch
			// 
			this.grpSearch.Controls.Add(this.pictureBox2);
			this.grpSearch.Controls.Add(this.btnSearch);
			this.grpSearch.Controls.Add(this.lblSearchString);
			this.grpSearch.Controls.Add(this.lvPatients);
			this.grpSearch.Controls.Add(this.txtSearchString);
			this.grpSearch.Dock = System.Windows.Forms.DockStyle.Left;
			this.grpSearch.Location = new System.Drawing.Point(56, 0);
			this.grpSearch.Name = "grpSearch";
			this.grpSearch.Size = new System.Drawing.Size(296, 400);
			this.grpSearch.TabIndex = 0;
			this.grpSearch.TabStop = false;
			this.grpSearch.Text = "Patient Search";
			// 
			// pictureBox2
			// 
			this.pictureBox2.Image = ((System.Drawing.Image)(resources.GetObject("pictureBox2.Image")));
			this.pictureBox2.Location = new System.Drawing.Point(8, 20);
			this.pictureBox2.Name = "pictureBox2";
			this.pictureBox2.Size = new System.Drawing.Size(20, 20);
			this.pictureBox2.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.pictureBox2.TabIndex = 114;
			this.pictureBox2.TabStop = false;
			// 
			// btnSearch
			// 
			this.btnSearch.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.btnSearch.EnabledButton = false;
			this.btnSearch.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.btnSearch.Image = ((System.Drawing.Image)(resources.GetObject("btnSearch.Image")));
			this.btnSearch.Location = new System.Drawing.Point(213, 38);
			this.btnSearch.Name = "btnSearch";
			this.btnSearch.Size = new System.Drawing.Size(75, 23);
			this.btnSearch.TabIndex = 2;
			this.toolTip1.SetToolTip(this.btnSearch, "Enter Full Name, Full Patient ID, Full Last Name, Last 4 Patient ID, or First Ini" +
				"tial of Last Name + Last 4 Patient ID.");
			this.btnSearch.Click += new System.EventHandler(this.btnSearch_Click);
			// 
			// lblSearchString
			// 
			this.lblSearchString.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lblSearchString.Location = new System.Drawing.Point(28, 12);
			this.lblSearchString.Name = "lblSearchString";
			this.lblSearchString.Size = new System.Drawing.Size(260, 24);
			this.lblSearchString.TabIndex = 0;
			this.lblSearchString.Text = "Last Name or Patient ID: Full, Last 4, or Last 5*";
			this.lblSearchString.TextAlign = System.Drawing.ContentAlignment.BottomLeft;
			// 
			// lvPatients
			// 
			this.lvPatients.AccessibleDescription = "Patients";
			this.lvPatients.AccessibleName = "Patients";
			this.lvPatients.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.lvPatients.AutoArrange = false;
			this.lvPatients.Columns.AddRange(new System.Windows.Forms.ColumnHeader[] {
																						 this.columnHeader1,
																						 this.columnHeader2,
																						 this.columnHeader3});
			this.lvPatients.FullRowSelect = true;
			this.lvPatients.HideSelection = false;
			this.lvPatients.Location = new System.Drawing.Point(8, 60);
			this.lvPatients.MultiSelect = false;
			this.lvPatients.Name = "lvPatients";
			this.lvPatients.PageLoadCount = 0;
			this.lvPatients.Size = new System.Drawing.Size(280, 332);
			this.lvPatients.TabIndex = 3;
			this.lvPatients.View = System.Windows.Forms.View.Details;
			this.lvPatients.DoubleClick += new System.EventHandler(this.lvPatients_DoubleClick);
			this.lvPatients.SelectedIndexChanged += new System.EventHandler(this.lvPatients_SelectedIndexChanged);
			this.lvPatients.ColumnClick += new System.Windows.Forms.ColumnClickEventHandler(this.lvPatients_ColumnClick);
			this.lvPatients.MoreDataRequired += new System.EventHandler(this.lvPatients_MoreDataRequired);
			// 
			// columnHeader1
			// 
			this.columnHeader1.Text = "Name";
			this.columnHeader1.Width = 150;
			// 
			// columnHeader2
			// 
			this.columnHeader2.Text = "Patient ID";
			this.columnHeader2.Width = 80;
			// 
			// columnHeader3
			// 
			this.columnHeader3.Text = "DOB";
			this.columnHeader3.Width = 80;
			// 
			// txtSearchString
			// 
			this.txtSearchString.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtSearchString.Barcode = false;
			this.txtSearchString.BarcodeType = gov.va.med.vbecs.Common.BarcodeType.None;
			this.txtSearchString.Location = new System.Drawing.Point(8, 40);
			this.txtSearchString.MaxLength = 30;
			this.txtSearchString.Name = "txtSearchString";
			this.txtSearchString.Size = new System.Drawing.Size(204, 20);
			this.txtSearchString.TabIndex = 1;
			this.txtSearchString.Text = "";
			this.txtSearchString.Leave += new System.EventHandler(this.txtSearchString_Leave);
			this.txtSearchString.TextChanged += new System.EventHandler(this.txtSearchString_TextChanged);
			// 
			// grpSelected
			// 
			this.grpSelected.Controls.Add(this.label1);
			this.grpSelected.Controls.Add(this.txtPatientLocation);
			this.grpSelected.Controls.Add(this.lblPercentage);
			this.grpSelected.Controls.Add(this.lblCompatibility);
			this.grpSelected.Controls.Add(this.txtCompatibbility);
			this.grpSelected.Controls.Add(this.picJustified);
			this.grpSelected.Controls.Add(this.patientInfoToolbar1);
			this.grpSelected.Controls.Add(this.specialInstructionsDisplayPanel1);
			this.grpSelected.Controls.Add(this.txtPatientName);
			this.grpSelected.Controls.Add(this.lblPatientName);
			this.grpSelected.Controls.Add(this.txtSSN);
			this.grpSelected.Controls.Add(this.txtABORh);
			this.grpSelected.Controls.Add(this.txtSex);
			this.grpSelected.Controls.Add(this.lblDOB);
			this.grpSelected.Controls.Add(this.txtDOB);
			this.grpSelected.Controls.Add(this.lblSSN);
			this.grpSelected.Controls.Add(this.lblSex);
			this.grpSelected.Controls.Add(this.lblABORh);
			this.grpSelected.Dock = System.Windows.Forms.DockStyle.Fill;
			this.grpSelected.Location = new System.Drawing.Point(356, 0);
			this.grpSelected.Name = "grpSelected";
			this.grpSelected.Size = new System.Drawing.Size(368, 400);
			this.grpSelected.TabIndex = 2;
			this.grpSelected.TabStop = false;
			this.grpSelected.Text = "Selected Patient";
			// 
			// label1
			// 
			this.label1.Location = new System.Drawing.Point(4, 88);
			this.label1.Name = "label1";
			this.label1.Size = new System.Drawing.Size(96, 23);
			this.label1.TabIndex = 73;
			this.label1.Text = "Patient Location";
			this.label1.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// txtPatientLocation
			// 
			this.txtPatientLocation.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtPatientLocation.Location = new System.Drawing.Point(104, 88);
			this.txtPatientLocation.Name = "txtPatientLocation";
			this.txtPatientLocation.ReadOnly = true;
			this.txtPatientLocation.Size = new System.Drawing.Size(256, 20);
			this.txtPatientLocation.TabIndex = 72;
			this.txtPatientLocation.TabStop = false;
			this.txtPatientLocation.Text = "";
			// 
			// lblPercentage
			// 
			this.lblPercentage.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.lblPercentage.Location = new System.Drawing.Point(192, 112);
			this.lblPercentage.Name = "lblPercentage";
			this.lblPercentage.Size = new System.Drawing.Size(15, 20);
			this.lblPercentage.TabIndex = 12;
			this.lblPercentage.Text = "%";
			this.lblPercentage.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// lblCompatibility
			// 
			this.lblCompatibility.Font = new System.Drawing.Font("Arial", 7.9F);
			this.lblCompatibility.Location = new System.Drawing.Point(4, 112);
			this.lblCompatibility.Name = "lblCompatibility";
			this.lblCompatibility.Size = new System.Drawing.Size(66, 20);
			this.lblCompatibility.TabIndex = 10;
			this.lblCompatibility.Text = "Compatibility";
			this.lblCompatibility.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// txtCompatibbility
			// 
			this.txtCompatibbility.AccessibleDescription = "Compatibility";
			this.txtCompatibbility.AccessibleName = "Compatibility";
			this.txtCompatibbility.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtCompatibbility.Location = new System.Drawing.Point(72, 112);
			this.txtCompatibbility.Name = "txtCompatibbility";
			this.txtCompatibbility.ReadOnly = true;
			this.txtCompatibbility.Size = new System.Drawing.Size(83, 20);
			this.txtCompatibbility.TabIndex = 9;
			this.txtCompatibbility.Text = "";
			this.txtCompatibbility.TextAlign = System.Windows.Forms.HorizontalAlignment.Right;
			// 
			// picJustified
			// 
			this.picJustified.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.picJustified.Image = ((System.Drawing.Image)(resources.GetObject("picJustified.Image")));
			this.picJustified.Location = new System.Drawing.Point(218, 66);
			this.picJustified.Name = "picJustified";
			this.picJustified.Size = new System.Drawing.Size(16, 16);
			this.picJustified.SizeMode = System.Windows.Forms.PictureBoxSizeMode.AutoSize;
			this.picJustified.TabIndex = 71;
			this.picJustified.TabStop = false;
			this.picJustified.Visible = false;
			// 
			// patientInfoToolbar1
			// 
			this.patientInfoToolbar1.AccessibleDescription = "Patient Information";
			this.patientInfoToolbar1.AccessibleName = "Patient Information";
			this.patientInfoToolbar1.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.patientInfoToolbar1.EnabledBloodAvailButton = false;
			this.patientInfoToolbar1.EnabledMedicationButton = false;
			this.patientInfoToolbar1.EnabledRecentOrdersButton = false;
			this.patientInfoToolbar1.EnabledRecentTransfusionsIssuesButton = false;
			this.patientInfoToolbar1.EnabledSIAndTRButton = false;
			this.patientInfoToolbar1.EnabledTransfusionReactionHistoryButton = false;
			this.patientInfoToolbar1.Location = new System.Drawing.Point(208, 112);
			this.patientInfoToolbar1.Name = "patientInfoToolbar1";
			this.patientInfoToolbar1.ParentFormInstanceGuid = new System.Guid("00000000-0000-0000-0000-000000000000");
			this.patientInfoToolbar1.Patient = null;
			this.patientInfoToolbar1.Size = new System.Drawing.Size(154, 20);
			this.patientInfoToolbar1.TabIndex = 10;
			// 
			// specialInstructionsDisplayPanel1
			// 
			this.specialInstructionsDisplayPanel1.AccessibleDescription = "Patient Special Instructions";
			this.specialInstructionsDisplayPanel1.AccessibleName = "Patient Special Instructions";
			this.specialInstructionsDisplayPanel1.Anchor = ((System.Windows.Forms.AnchorStyles)((((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Bottom) 
				| System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.specialInstructionsDisplayPanel1.Font = new System.Drawing.Font("Arial", 6.75F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.specialInstructionsDisplayPanel1.Location = new System.Drawing.Point(8, 144);
			this.specialInstructionsDisplayPanel1.Name = "specialInstructionsDisplayPanel1";
			this.specialInstructionsDisplayPanel1.Patient = null;
			this.specialInstructionsDisplayPanel1.Size = new System.Drawing.Size(350, 240);
			this.specialInstructionsDisplayPanel1.TabIndex = 11;
			// 
			// txtPatientName
			// 
			this.txtPatientName.AccessibleDescription = "Name";
			this.txtPatientName.AccessibleName = "Name";
			this.txtPatientName.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtPatientName.Location = new System.Drawing.Point(69, 16);
			this.txtPatientName.Name = "txtPatientName";
			this.txtPatientName.ReadOnly = true;
			this.txtPatientName.Size = new System.Drawing.Size(291, 20);
			this.txtPatientName.TabIndex = 4;
			this.txtPatientName.Text = "";
			// 
			// lblPatientName
			// 
			this.lblPatientName.Location = new System.Drawing.Point(4, 16);
			this.lblPatientName.Name = "lblPatientName";
			this.lblPatientName.Size = new System.Drawing.Size(40, 20);
			this.lblPatientName.TabIndex = 0;
			this.lblPatientName.Text = "Name";
			this.lblPatientName.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// txtSSN
			// 
			this.txtSSN.AccessibleDescription = "Patient I D";
			this.txtSSN.AccessibleName = "Patient I D";
			this.txtSSN.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtSSN.Location = new System.Drawing.Point(69, 40);
			this.txtSSN.Name = "txtSSN";
			this.txtSSN.ReadOnly = true;
			this.txtSSN.Size = new System.Drawing.Size(166, 20);
			this.txtSSN.TabIndex = 5;
			this.txtSSN.Text = "";
			// 
			// txtABORh
			// 
			this.txtABORh.AccessibleDescription = "A B O R h";
			this.txtABORh.AccessibleName = "A B O R h";
			this.txtABORh.Anchor = ((System.Windows.Forms.AnchorStyles)(((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Left) 
				| System.Windows.Forms.AnchorStyles.Right)));
			this.txtABORh.Location = new System.Drawing.Point(69, 64);
			this.txtABORh.Name = "txtABORh";
			this.txtABORh.ReadOnly = true;
			this.txtABORh.Size = new System.Drawing.Size(166, 20);
			this.txtABORh.TabIndex = 7;
			this.txtABORh.Text = "";
			this.txtABORh.MouseHover += new System.EventHandler(this.txtABORh_MouseHover);
			// 
			// txtSex
			// 
			this.txtSex.AccessibleDescription = "Sex";
			this.txtSex.AccessibleName = "Sex";
			this.txtSex.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.txtSex.Location = new System.Drawing.Point(280, 64);
			this.txtSex.Name = "txtSex";
			this.txtSex.ReadOnly = true;
			this.txtSex.Size = new System.Drawing.Size(80, 20);
			this.txtSex.TabIndex = 8;
			this.txtSex.Text = "";
			// 
			// lblDOB
			// 
			this.lblDOB.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.lblDOB.Location = new System.Drawing.Point(246, 40);
			this.lblDOB.Name = "lblDOB";
			this.lblDOB.Size = new System.Drawing.Size(35, 20);
			this.lblDOB.TabIndex = 4;
			this.lblDOB.Text = "DOB";
			this.lblDOB.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// txtDOB
			// 
			this.txtDOB.AccessibleDescription = "Date of Birth";
			this.txtDOB.AccessibleName = "Date of Birth";
			this.txtDOB.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.txtDOB.Location = new System.Drawing.Point(280, 40);
			this.txtDOB.Name = "txtDOB";
			this.txtDOB.ReadOnly = true;
			this.txtDOB.Size = new System.Drawing.Size(80, 20);
			this.txtDOB.TabIndex = 6;
			this.txtDOB.Text = "";
			// 
			// lblSSN
			// 
			this.lblSSN.Location = new System.Drawing.Point(4, 40);
			this.lblSSN.Name = "lblSSN";
			this.lblSSN.Size = new System.Drawing.Size(84, 20);
			this.lblSSN.TabIndex = 2;
			this.lblSSN.Text = "Patient ID";
			this.lblSSN.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// lblSex
			// 
			this.lblSex.Anchor = ((System.Windows.Forms.AnchorStyles)((System.Windows.Forms.AnchorStyles.Top | System.Windows.Forms.AnchorStyles.Right)));
			this.lblSex.Location = new System.Drawing.Point(246, 64);
			this.lblSex.Name = "lblSex";
			this.lblSex.Size = new System.Drawing.Size(36, 20);
			this.lblSex.TabIndex = 8;
			this.lblSex.Text = "Sex";
			this.lblSex.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// lblABORh
			// 
			this.lblABORh.Location = new System.Drawing.Point(4, 64);
			this.lblABORh.Name = "lblABORh";
			this.lblABORh.Size = new System.Drawing.Size(68, 20);
			this.lblABORh.TabIndex = 6;
			this.lblABORh.Text = "ABO/Rh";
			this.lblABORh.TextAlign = System.Drawing.ContentAlignment.MiddleLeft;
			// 
			// toolTip1
			// 
			this.toolTip1.ShowAlways = true;
			// 
			// splitter1
			// 
			this.splitter1.Location = new System.Drawing.Point(352, 0);
			this.splitter1.MinExtra = 275;
			this.splitter1.MinSize = 275;
			this.splitter1.Name = "splitter1";
			this.splitter1.Size = new System.Drawing.Size(4, 400);
			this.splitter1.TabIndex = 1;
			this.splitter1.TabStop = false;
			// 
			// panel1
			// 
			this.panel1.Dock = System.Windows.Forms.DockStyle.Left;
			this.panel1.Location = new System.Drawing.Point(52, 0);
			this.panel1.Name = "panel1";
			this.panel1.Size = new System.Drawing.Size(4, 400);
			this.panel1.TabIndex = 4;
			// 
			// PatientSelectTool
			// 
			this.Controls.Add(this.grpSelected);
			this.Controls.Add(this.splitter1);
			this.Controls.Add(this.grpSearch);
			this.Controls.Add(this.panel1);
			this.Controls.Add(this.grpImage);
			this.Font = new System.Drawing.Font("Arial", 8.25F, System.Drawing.FontStyle.Regular, System.Drawing.GraphicsUnit.Point, ((System.Byte)(0)));
			this.hlpHelpChm.SetHelpKeyword(this,"Patient_Selection.htm");
            this.hlpHelpChm.SetHelpNavigator(this, System.Windows.Forms.HelpNavigator.Topic);
			this.Name = "PatientSelectTool";
			this.hlpHelpChm.SetShowHelp(this, true);
			this.Size = new System.Drawing.Size(724, 400);
			this.grpImage.ResumeLayout(false);
			this.grpSearch.ResumeLayout(false);
			this.grpSelected.ResumeLayout(false);
			this.ResumeLayout(false);

		}
		#endregion

		#region Constructor / Dispose

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/17/2002</CreationDate>
		/// <summary>
		/// Patient select tool class constructor.
		/// </summary>
		public PatientSelectTool()
		{
			InitializeComponent();
			//
			this.patientInfoToolbar1.EnableAllButtons(false);
			//
			// Must be inside try/catch block, because it fails when component used in design time
			try
			{
				patient = new BOL.Patient();
			}
			catch
			{
			}
			//
			_uidEx = Common.RegularExpressions.SpecimenUid();
			_lastfiveEx = Common.RegularExpressions.PatientInitialAndLastFourSSN();
			_ssnEx = Common.RegularExpressions.PatientSsn();
			_nameEx = Common.RegularExpressions.PatientName();
			//
			this.lvPatients.Items.Clear();
			this.lvPatients.PageLoadCount = 50;
			//
			this.btnSearch.Text = "&Search";
			//
			this.ResetToolTips(); // CR 3028
            //Defect 210321 
            // 
            // hlphelpChm
            // 
            this.hlpHelpChm.HelpNamespace = Environment.CurrentDirectory + "\\VBECS.chm";
		}

		/// <summary> 
		/// Clean up any resources being used.
		/// </summary>
		protected override void Dispose( bool disposing )
		{
			if( disposing )
			{
				if(components != null)
				{
					components.Dispose();
				}
			}
			base.Dispose( disposing );
		}

		#endregion

		#region Properties

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Override of Anchor property
		/// </summary>
		[
		Category("Layout"),
		Description("The anchor of the control"),
		Bindable(false),
		Browsable(true),
		]
		public override AnchorStyles Anchor
		{
			get { return base.Anchor; }
			set { base.Anchor = value; }
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		/// <summary>
		/// Gets the patient selected via the user control
		/// </summary>
		public Patient SelectedPatient
		{
			get { return patient; }
		}

		/// <summary>
		/// Get/Set SearchVistaLink
		/// </summary>
		public bool SearchVistaLink
		{
			get 
			{
				return this._searchVistALink;
			}
			set
			{
				this._searchVistALink = value;
			}
		}

		/// <summary>
		/// Determines whether the SI and TR panel should be visible
		/// </summary>
		[
		Category("Appearance"),
		Description("Determines whether the SI and TR panel should be visible"),
		Bindable(false),
		Browsable(true),
		]
		public bool DisplaySIAndTRs
		{
			get
			{
				return this.specialInstructionsDisplayPanel1.Visible;
			}
			set
			{
				this.specialInstructionsDisplayPanel1.Visible = value;
			}
		}

		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to call business layer to search for patients matching given search 
		/// criteria and display results in list view.
		/// BR_73.21, BR_73.22
		/// </summary>
		/// <param name="attemptToRestoreVistALinkIfItsDown"></param>
		/// <param name="patientName"></param>
		/// <param name="patientSSN"></param>
		/// <param name="specimenUid"></param>
		private void PatientSearch( bool attemptToRestoreVistALinkIfItsDown, string patientName, string patientSSN, string specimenUid )
		{
			using (new WaitCursor())
			{
				patient = new BOL.Patient();
				//
				lvPatients.Items.Clear();
				// 
				_vbecsPatients = BOL.Patient.Search( patientName,patientSSN,specimenUid,Common.LogonUser.LogonUserDivisionCode );
				_vistaPatients = new DataTable();
				//		
				// In some use cases we also have to look to VistALink
				if( this.SearchVistaLink && IsVistaLinkAvailable( attemptToRestoreVistALinkIfItsDown ) )
				{	
					try
					{
						_vistaPatients = BOL.Patient.SearchVistALink( patientName, patientSSN, specimenUid );

						for(int i=0; i<_vistaPatients.Rows.Count; i++)
						{
							DataRow dr = _vistaPatients.Rows[i];

							if(_vbecsPatients.Columns.Contains(Common.VbecsTables.Patient.VistaPatientId) &&
								_vistaPatients.Columns.Contains((Common.VbecsTables.Patient.VistaPatientId)))
							{
								System.Data.DataRow [] dtRows = _vbecsPatients.Select( "VistaPatientId = " + dr[Common.VbecsTables.Patient.VistaPatientId] );
						
								if(dtRows.Length > 0)
								{
									if( _vistaPatients.Columns.Contains( TABLE.Patient.PatientFirstName ) )
									{
										_vistaPatients.Rows.Remove(dr);
										i--;
									}
								}
							}
						}

						//CR 2606 - removing patient if first name or last name is blank
						for(int i=0; i<_vistaPatients.Rows.Count; i++)
						{
							DataRow dr = _vistaPatients.Rows[i];
							if (_vistaPatients.Rows[i].IsNull(TABLE.Patient.PatientFirstName) == true)
							{
								GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.PatientNameTooLong());
								_vistaPatients.Rows.Remove(dr);
								i--;
								continue;
							}
							else if (_vistaPatients.Rows[i][ TABLE.Patient.PatientFirstName].ToString().Trim() == "")
							{
								GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.PatientNameTooLong());
								_vistaPatients.Rows.Remove(dr);
								i--;
								continue;
							}

							if (_vistaPatients.Rows[i].IsNull(TABLE.Patient.PatientLastName) == true)
							{
								GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.PatientNameTooLong());
								_vistaPatients.Rows.Remove(dr);
								i--;
								continue;
							}
							else if (_vistaPatients.Rows[i][ TABLE.Patient.PatientLastName].ToString().Trim() == "")
							{
								GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.PatientNameTooLong());
								_vistaPatients.Rows.Remove(dr);
								i--;
								continue;
							}
						}
					}
					catch(DAL.VistALink.OpenLibrary.VistALinkException ex)
					{
						GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.VistaLinkDataAcessFailure(ex));
					}
				}
			}
			//
			AddPageOfPatients();
		}

		/// <summary>
		/// AddPageOfPatients
		/// </summary>
		private void AddPageOfPatients()
		{
			int start = this.lvPatients.Items.Count;
			int end = ((start + this.lvPatients.PageLoadCount) < (_vbecsPatients.Rows.Count + _vistaPatients.Rows.Count)) ? (start + this.lvPatients.PageLoadCount - 1) : (_vbecsPatients.Rows.Count + _vistaPatients.Rows.Count - 1);
			//
			if (end >= start)
			{
				using (new WaitCursor())
				{
					this.lvPatients.BeginUpdate();
					for (int i = start; i <= end; i++)
					{
						if (i < _vbecsPatients.Rows.Count)
						{
							this.PopulateListView( this._vbecsPatients.Rows[i], false );
						}
						else
						{
							this.PopulateListView( this._vistaPatients.Rows[i-_vbecsPatients.Rows.Count], true);
						}
					}
					this.lvPatients.EndUpdate();

					// Only default selection if one match, else user must choose
					if ( lvPatients.Items.Count == 1 )
					{
						lvPatients.Items[0].Selected = true;
					}
					else
					{
						lvPatients.SelectedItems.Clear();
					}
					//
					controls.VbecsFunction.SetColumnWidths(this.lvPatients);
				}
			}
			if( lvPatients.Items.Count == 0 )
			{
				this.ClearPatientDetails();
				this.ResetToolTips(); // CR 3028
				this.lvPatients.Items.Add(new ListViewItem(Common.StrRes.InfoMsg.UC020.NoMatchingEntries().ResString));
				//
				controls.VbecsFunction.SetColumnWidths(this.lvPatients);
			}
		}

		/// <summary>
		/// IsVistaLinkAvailable
		/// </summary>
		private bool IsVistaLinkAvailable( bool attemptToRestore )
		{
			if( VistALink.CheckStatus() )
				return true;
			
			if( !attemptToRestore )
				return false;

			if( VistALink.EnsureAvailabilityWithoutFailureNotification() )
				return true;

			GuiMessenger.ShowMessageBox( Common.StrRes.InfoMsg.Common.VistALinkIsNotSearchedForPatient() );

			return false;
		}

		/// <summary>
		/// PopulateListView
		/// </summary>
		private void PopulateListView(System.Data.DataRow dr, bool importedFromVista)
		{
			ListViewItem lvPatient = new ListViewItem();
			lvPatient.Text = BOL.Patient.BuildDisplayName(dr[TABLE.Patient.PatientFirstName].ToString(),
				dr[TABLE.Patient.PatientMiddleName].ToString(),
				dr[TABLE.Patient.PatientLastName].ToString());

			if (dr.Table.Columns.Contains(TABLE.Patient.DisplayVistaPatientId))
			{
				lvPatient.SubItems.Add(dr[TABLE.Patient.DisplayVistaPatientId].ToString());
			}
			else
			{
				if (dr.IsNull(TABLE.Patient.PatientSsn))
				{
					lvPatient.SubItems.Add("");
				}
				else if (dr[TABLE.Patient.PatientSsn].ToString().Trim().Length<9)
				{
					lvPatient.SubItems.Add(dr[TABLE.Patient.PatientSsn].ToString());
				}
				else
				{
					lvPatient.SubItems.Add(dr[TABLE.Patient.PatientSsn].ToString().Insert(3,"-").Insert(6,"-"));
				}
			}

			if (dr.IsNull(TABLE.Patient.PatientDob))
			{
				lvPatient.SubItems.Add("");
			}
			else
			{
				lvPatient.SubItems.Add(Common.VBECSDateTime.FormatDateString( (System.DateTime) dr[TABLE.Patient.PatientDob] ));
			}

			//CR 2606 - added exception handling if patient data are incorrect
			try
			{
				BOL.Patient pat = new BOL.Patient(dr, importedFromVista);
				if (pat.DisplayName.Length > 31)
				{
					GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.PatientNameTooLong());
					return;
				}
				//
				lvPatient.Tag = dr;
				lvPatients.Items.Add(lvPatient);
			}
			catch (BOL.BusinessObjectException e)
			{
				GuiMessenger.ShowMessageBox(Common.StrRes.SysErrMsg.Common.Dynamic(e.Message));
				return;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to parse the entered search string to determine if name, ssn or accession 
		/// number entered.  Name, SSN and Accession Number are returned as out parameters.
		/// </summary>
		/// <param name="searchString"></param>
		/// <param name="patientName"></param>
		/// <param name="patientSSN"></param>
		/// <param name="specimenUid"></param>
		private void ParseSearchString(string searchString, out string patientName, out string patientSSN, out string specimenUid)
		{
			// Need to parse search string to determine if search is intended
			// to be on Name, full SSN, Last 4 or Last 5.
			//
			searchString = searchString.ToUpper();
			patientName = string.Empty;
			patientSSN = string.Empty;
			specimenUid = string.Empty;
			//
			// Check for UID
			if (_uidEx.IsMatch(searchString))
			{
				specimenUid = searchString;
				return;
			}
			// Check for Last 5
			if (_lastfiveEx.IsMatch(searchString))
			{
				patientName = searchString.Substring(0,1);
				patientSSN = searchString.Substring(1,4);
				return;
			}
			// Check for SSN
			if (_ssnEx.IsMatch(searchString))
			{
				patientSSN = searchString;
				return;
			}
			// Check for Name
			if (_nameEx.IsMatch(searchString))
			{
				patientName = searchString;
				return;
			}
		}

		/// <summary>
		/// CR 2472
		/// </summary>
		private void ClearPatientDetails()
		{
			this.txtABORh.Text = string.Empty;
			this.txtDOB.Text = string.Empty;
			this.txtPatientName.Text = string.Empty;
			this.txtSSN.Text = string.Empty;
			this.txtSex.Text = string.Empty;
			this.txtPatientLocation.Text = string.Empty;
			this.txtCompatibbility.Text = string.Empty;
			this.patientInfoToolbar1.Enabled = false;
			this.specialInstructionsDisplayPanel1.Patient = null;
			this.picJustified.Visible = false;
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Private method to display details of the selected patient in the user control.
		/// </summary>
		private void DisplaySelectedPatient()
		{
			AboRh patientAboRh = Patient.GetPatientAboRh(this.patient.PatientGuid);
			//
			if (patientAboRh.Abo != Common.ABO.NA)
			{
				this.txtABORh.Text = patientAboRh.AboRHText;
			}
			else
			{
				this.txtABORh.Text = "NR";
			}
			//
            this.txtDOB.Text = patient.DisplayDateOfBirth;
			//
			this.txtPatientName.Text = patient.DisplayName;
			//
			if (patient.DisplayPatientID != null && patient.DisplayPatientID.Length > 0)
			{
				this.txtSSN.Text = patient.DisplayPatientID;
			}
			else
			{
				if (patient.SSN.Length > 8)
				{
					this.txtSSN.Text = patient.SSN.Insert(3, "-").Insert(6, "-");
				}
				else
				{
					this.txtSSN.Text = "";
				}
			}
			//
			if (patient.Sex != Common.Sex.Unknown)
			{
				this.txtSex.Text = patient.Sex.ToString();
			}
			else
			{
				this.txtSex.Text = "";
			}
			//
			// Compatibility percentage
			this.txtCompatibbility.Text = patient.CompatibilityPercentage;
			//
			this.patientInfoToolbar1.Enabled = true;
			this.specialInstructionsDisplayPanel1.Patient = this.patient;
			this.picJustified.Visible = (this.patient.JustifiedAboRH);
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/19/2004</CreationDate>
		///
		/// <summary>
		/// Parses search string, stops timer and searches for patient using text 
		/// in textbox as the search string.
		/// </summary>
		/// <param name="restoreVistALinkIfItsDown">
		/// Specifies whether there should be an attempt to restore VistALink 
		/// connection if VistALink needs to be searched and it's down.
		/// </param>
		private void PrepareInputDataAndRunSearch( bool restoreVistALinkIfItsDown )
		{
			string patientName = string.Empty;
			string patientSSN = string.Empty;
			string specimenUid = string.Empty;
			
			this.ParseSearchString( txtSearchString.Text, out patientName, out patientSSN, out specimenUid );
			PatientSearch( restoreVistALinkIfItsDown, patientName, patientSSN, specimenUid );
		}

		/// <summary>
		/// 
		/// </summary>
		public System.Data.DataRow SelectedPatientDataRow()
		{
			return _dataRow;				
		}

		/// <summary>
		/// CR 3028
		/// </summary>
		private void ResetToolTips()
		{
            this.toolTip1.RemoveAll(); //CR 3282 erasing all tooltips before resetting them all again
			this.toolTip1.SetToolTip(this.btnSearch, TOOLTIP_BTN_SEARCH);
			this.toolTip1.SetToolTip(this.lvPatients, TOOLTIP_LV_PATIENTS);
			this.toolTip1.SetToolTip(this.txtSearchString, TOOLTIP_TXT_SEARCH_STRING);
            this.toolTip1.SetToolTip(this.picJustified, Common.StrRes.Tooltips.Patient.HistoryOfJustifiedAboRhChange().ResString); //CR 3033
            this.toolTip1.SetToolTip(this.txtPatientName, Common.StrRes.Tooltips.Patient.NameToolTip().ResString);
            this.toolTip1.SetToolTip(this.txtSSN, Common.StrRes.Tooltips.Patient.SsnToolTip().ResString);
            this.toolTip1.SetToolTip(this.txtABORh, Common.StrRes.Tooltips.Patient.AboRhToolTip().ResString);
            this.toolTip1.SetToolTip(this.txtSex, Common.StrRes.Tooltips.Patient.SexToolTip().ResString);
            this.toolTip1.SetToolTip(this.txtDOB, Common.StrRes.Tooltips.Patient.DateOfBirthToolTip().ResString);
			this.toolTip1.SetToolTip(this.txtSearchString, TOOLTIP_TXT_SEARCH_STRING);
			this.toolTip1.SetToolTip(this.txtPatientLocation, PatientLocation.LOCATION_TOOLTIP_BASE);
		}
	
		#endregion

		#region Events

		/// <summary>
		/// ABO\Rh TextBox Move Hover Event Handler
		/// </summary>
		private void txtABORh_MouseHover(object sender, System.EventArgs e)
		{
			// CR 3028: only show details if patient is selected.
			if ( this.patient != null && this.patient.PatientGuid != Guid.Empty )
			{
                this.toolTip1.SetToolTip(this.txtABORh, Common.StrRes.Tooltips.Patient.AboRhToolTip().ResString + NEWLINE + this.patient.GetPatientHistoricABORhForTooltip());
			}
		}

		/// <summary>
		/// ListView Patients MoreDataRequired Event Handler
		/// </summary>
		private void lvPatients_MoreDataRequired(object sender, EventArgs e)
		{
			this.AddPageOfPatients();
		}

		/// <summary>
		/// Event that handles the click on special instructions and transfusion requirements button
		/// </summary>
		public event EventHandler SIAndTRClick
		{
			add
			{
				onSIAndTRClick += value;
			}
			remove
			{
				onSIAndTRClick -= value;
			}
		}

		/// <summary>
		/// Event that handles click on the blood availability button
		/// </summary>
		public event EventHandler BloodAvailClick
		{
			add
			{
				onBloodAvailClick += value;
			}
			remove
			{
				onBloodAvailClick -= value;
			}
		}

		/// <summary>
		/// Event that handles click on the display patient VBECS data
		/// </summary>
		public event EventHandler VbecsInfoClick
		{
			add
			{
				onVbecsInfoClick += value;
			}
			remove
			{
				onVbecsInfoClick -= value;
			}
		}

		/// <summary>
		/// Event that handles click on the patient medications button
		/// </summary>
		public event EventHandler MedicationsClick
		{
			add
			{
				onMedicationsClick += value;
			}
			remove
			{
				onMedicationsClick -= value;
			}
		}

		/// <summary>
		/// Event that handles click on the transfusion history button
		/// </summary>
		public event EventHandler TransHistoryClick
		{
			add
			{
				onTransHistoryClick += value;
			}
			remove
			{
				onTransHistoryClick -= value;
			}
		}

		/// <summary>
		/// Event that handles click on the patient clinical data button
		/// </summary>
		public event EventHandler ClinicalInfoClick
		{
			add
			{
				onClinicalInfoClick += value;
			}
			remove
			{
				onClinicalInfoClick -= value;
			}
		}

		/// <summary>
		/// Calls the event when special instructions and patient transfusion requirements button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnSpecialInstructions_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onSIAndTRClick != null)
			{
				onSIAndTRClick(sender,e);
			}
		}

		/// <summary>
		/// Calls the event when blood availability button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnBloodAvailability_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onBloodAvailClick != null)
			{
				onBloodAvailClick(sender,e);
			}
		}

		/// <summary>
		/// Calls the event when patient VBECS data button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnPatientInfo_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onVbecsInfoClick != null)
			{
				onVbecsInfoClick(sender,e);
			}
		}

		/// <summary>
		/// Calls the event when patient current medications button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnCurrentMeds_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onMedicationsClick != null)
			{
				onMedicationsClick(sender,e);
			}
		}

		/// <summary>
		/// Calls the event when patient transfusion history button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnTransfusionHistory_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onTransHistoryClick != null)
			{
				onTransHistoryClick(sender,e);
			}
		}

		/// <summary>
		/// Calls the event when patient clinical data button is being clicked
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnClinicalData_Click(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (onClinicalInfoClick != null)
			{
				onClinicalInfoClick(sender,e);
			}
		}

		/// <summary>
		/// Indicates whether the SI (and) TR button is enabled
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the SI & TR button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledSIAndTRButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledSIAndTRButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledSIAndTRButton = value;
			}
		}

		/// <summary>
		/// Indicates whether the blood availability button is enabled"
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the blood availability button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledBloodAvailButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledBloodAvailButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledBloodAvailButton = value;
			}
		}

		/// <summary>
		/// Indicates whether the Recent Orders button is enabled
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the Recent Orders button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledRecentOrdersButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledRecentOrdersButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledRecentOrdersButton = value;
			}
		}

		/// <summary>
		/// Indicates whether the Medication Profile button is enabled
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the Medication Profile button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledMedicationButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledMedicationButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledMedicationButton = value;
			}
		}

		/// <summary>
		/// Indicates whether the Recent Transfusions/Issues  button is enabled
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the Recent Transfusions/Issues  button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledRecentTransfusionsIssuesButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledRecentTransfusionsIssuesButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledRecentTransfusionsIssuesButton = value;
			}
		}

		/// <summary>
		/// Indicates whether the Transfusion Recation History button is enabled
		/// </summary>
		[
		Category("Behavior"),
		Description("Indicates whether the Transfusion Recation History button is enabled"),
		DefaultValue(true),
		Bindable(false),
		Browsable(true),
		]
		public bool EnabledTransfusionReactionHistoryButton
		{
			get
			{
				return this.patientInfoToolbar1.EnabledTransfusionReactionHistoryButton;
			}
			set
			{
				this.patientInfoToolbar1.EnabledTransfusionReactionHistoryButton = value;
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Textbox TextChanged event. Determines if details entered so far merit enabling search timer.
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void txtSearchString_TextChanged(object sender, System.EventArgs e)
		{
			this.OnUserActivity();

			if (this.txtSearchString.Barcode)
			{
				btnSearch.EnabledButton = false;
			}
			else
			{
				OnPatientDeselected();
				// 
				string searchStr = txtSearchString.Text.ToUpper();
				//
				if (_uidEx.IsMatch(searchStr) || _lastfiveEx.IsMatch(searchStr) || _ssnEx.IsMatch(searchStr) || _nameEx.IsMatch(searchStr))
				{
                    //CR 3467
				    if (_ssnEx.IsMatch(searchStr) && (searchStr.Length > 4 && searchStr.Length < 9))
				    {
				        btnSearch.EnabledButton = false;
				    }
                    //Entered Last Initial + 9 digits.  This shouldnt enable Search.  S234560050
                    else if (_uidEx.IsMatch(searchStr) && (searchStr.Length > 5 && searchStr.Length < 11) && !_nameEx.IsMatch(searchStr))
                    {
                        btnSearch.EnabledButton = false;
                    }
                    else
                    {
                        btnSearch.EnabledButton = true;
                    }
				}
				else
				{
					btnSearch.EnabledButton = false;
				}
			}
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>08/12/2004</CreationDate>
		///
		/// <summary>
		/// Leave event
		/// </summary>
		private void txtSearchString_Leave(object sender, System.EventArgs e)
		{
			this.OnUserActivity();
			//
			if (this.txtSearchString.Barcode)
			{
				string[] pieces = this.txtSearchString.Text.Split(',');
				//
				if (pieces.Length>0)
				{
					this.txtSearchString.Text = pieces[0].Replace("-","");
					if (_ssnEx.IsMatch(this.txtSearchString.Text))
					{
						btnSearch.EnabledButton = true;
						PatientSearch(false,"",this.txtSearchString.Text,"");
					}
					else
					{
						btnSearch.EnabledButton = false;
						this.txtSearchString.Barcode = false;
					}
				}
				else
				{
					this.txtSearchString.Text = "";
					this.txtSearchString.Barcode = false;
				}
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Raises PatientSelected event
		/// </summary>
		protected virtual void OnPatientSelected()
		{
			using (new WaitCursor())
			{
				this.patientInfoToolbar1.Patient = this.patient;
				// No need to enable this button, beacuse SIs and TRs are displayed on the control
				this.patientInfoToolbar1.EnabledSIAndTRButton = false;
				if (PatientSelected!=null) 
					PatientSelected(null,EventArgs.Empty);
			}
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Raises PatientDeselected event
		/// </summary>
		protected virtual void OnPatientDeselected()
		{
			this.patientInfoToolbar1.Patient = null;

			if( PatientDeselected != null ) 
				PatientDeselected( null, EventArgs.Empty );
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Traps listview's DoubleClick event andrRaises user control's DoubleClick event
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void lvPatients_DoubleClick(object sender, System.EventArgs e)
		{
			if(( lvPatients.SelectedItems.Count > 0 )&&(lvPatients.SelectedItems[0].Tag != null))
				OnDoubleClick(e);
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/13/2002</CreationDate>
		///
		/// <summary>
		/// Listview SelectedIndexChanged event. Sets SelectedPatient to patient selected in listview and displays details
		/// CR 2472
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void lvPatients_SelectedIndexChanged(object sender, System.EventArgs e)
		{
            if (this.Parent == null) return;        //LOCKING 2.0 - for some reason NOW this event fires if the form is closed by code(UC43)

			this.OnUserActivity();
			//
			try
			{
				if (lvPatients.SelectedItems.Count > 0)
				{
					if (lvPatients.SelectedItems[0].Tag == null)
					{
						// User selected the no records found row!
						this.ClearPatientDetails();
						this.ResetToolTips(); // CR 3028
						OnPatientDeselected();
						return;
					}
					//
					this._dataRow = (DataRow)lvPatients.SelectedItems[0].Tag;
					this.patient = new Patient(_dataRow);
					this.patient.AboRh = BOL.Patient.GetPatientAboRh(this.patient.PatientGuid);
					//
					// CR 2472 BEGIN
					StringBuilder locationBuilder = new StringBuilder();
					//
					PatientLocation patientLocation = PatientLocation.GetCurrentPatientLocation(this.patient.PatientGuid);
                    this.txtPatientLocation.Text = PatientLocation.GetName(patientLocation);
                    this.toolTip1.SetToolTip(this.txtPatientLocation, PatientLocation.GetToolTip(patientLocation));

					//
					if (this.patient.IsPatientDeceased)
					{
						// BR_1.67
						if (GuiMessenger.ShowMessageBox(Common.StrRes.ConfMsg.Common.SelectDeceasedPatient(this.patient.DisplayName))!= DialogResult.Yes)
						{
							this.ClearPatientDetails();
							this.ResetToolTips(); // CR 3028
							OnPatientDeselected();
							return;
						}
					}
					//
					// If the patient search was originally on accession number, set SpecimenUid property
					if (_uidEx.IsMatch(txtSearchString.Text.ToUpper()))
					{
						this.patient.SpecimenUid = txtSearchString.Text.ToUpper();
					}
					//
					DisplaySelectedPatient();
					// Initiates OnPatientSelected event
					OnPatientSelected();
				}
				else
				{
					this.ClearPatientDetails();
					this.ResetToolTips(); // CR 3028
					OnPatientDeselected();
				}
			}
			catch (Exception err)
			{
				MessageBox.Show(err.Message,"VBECS - Error",MessageBoxButtons.OK,MessageBoxIcon.Exclamation);
			}
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>06/19/2004</CreationDate>
		///
		/// <summary>
		/// Executes patient search. Always tries to restore VistALink connection if it's down.
		/// CR 2472 
		/// </summary>
		/// <param name="sender"></param>
		/// <param name="e"></param>
		private void btnSearch_Click( object sender, System.EventArgs e )
		{
			// CR 2472
			this.ClearPatientDetails();
			this.ResetToolTips(); // CR 3028
			//
			this.OnUserActivity();
			//
			PrepareInputDataAndRunSearch( true );
			//
			// CR 2472
			this.columnHeader1.Width = 150;
			this.columnHeader2.Width = 80;
			this.columnHeader3.Width = 80;
		}

		/// <summary>
		/// ListView Patients Colum Click Event Handler
		/// </summary>
		private void lvPatients_ColumnClick(object sender, System.Windows.Forms.ColumnClickEventArgs e)
		{
			this.OnUserActivity();

			if (e.Column == 2)
			{
				ColumnClickSorter.ColumnClickHandler(sender, e, lvPatients.CheckBoxes, ColumnClickSorter.ColumnSortType.DateTime);
			}
			else 
			{
				ColumnClickSorter.ColumnClickHandler(sender, e, lvPatients.CheckBoxes, ColumnClickSorter.ColumnSortType.String);
			}
		}

		/// <summary>
		/// **** 508 Compliance ***
		/// </summary>
		/// <returns></returns>
		public string GetErrorMessagesFor508()
		{
			string textToSpeak = string.Empty;
			if(this.picJustified.Visible)
			{
				textToSpeak = "The patient has a history of  justified ABO/Rh change";
			}

			if (this.btnSearch.EnabledButton == false)
			{
				textToSpeak = string.Concat(textToSpeak, "\n", this.toolTip1.GetToolTip(this.btnSearch));
				return textToSpeak;
			}
			else
			{
				return textToSpeak;
			}
		}
		#endregion		
	}
}

